//This is the original code. Can you post complete code that works in terms of decryption?
import React, { useState, useEffect, useCallback } from "react";
import {
  Authenticator,
  Button,
  TextField,
  Heading,
  Flex,
  View,
  Divider,
  Table,
  TableHead,
  TableRow,
  TableCell,
  TableBody,
} from "@aws-amplify/ui-react";
import { Amplify } from "aws-amplify";
import "@aws-amplify/ui-react/styles.css";
import { generateClient } from "aws-amplify/data";
import outputs from "../amplify_outputs.json";
import { getCurrentUser } from "aws-amplify/auth";
import sjcl from "sjcl";

// Enable backend interactions.
Amplify.configure(outputs);
const client = generateClient({ authMode: "userPool" });

const AppLayout = ({ children }) => (
  <Flex
    className="App"
    justifyContent="center"
    alignItems="center"
    direction="column"
    width="70%"
    margin="0 auto"
  >
    {children}
  </Flex>
);
function safeDecrypt(userId, encrypted) {
  try {
    if (!userId || !encrypted) return null;
    return sjcl.decrypt(userId, encrypted);
  } catch (err) {
    console.error("Decryption failed:", err);
    return "[decryption failed]";
  }
}

const RecordsTable = ({ filteredRecords, userId }) => (
  <Table>
    <TableHead>
      <TableRow>
        <TableCell>Login</TableCell>
        <TableCell>Date</TableCell>
        <TableCell>Category</TableCell>
        <TableCell>Encrypted</TableCell>
      </TableRow>
    </TableHead>
    <TableBody>
      {filteredRecords.map((record) => (
        <TableRow key={record.id || record.category}>
          <TableCell>{record.login}</TableCell>
          <TableCell>{record.date}</TableCell>
          <TableCell>{record.category}</TableCell>
          <TableCell>{safeDecrypt(userId, record.encrypted)}</TableCell>
        </TableRow>
      ))}
    </TableBody>
  </Table>
);

const SearchForm = ({ searchQuery, setSearchQuery, handleSearch }) => (
  <View as="form" onSubmit={handleSearch}>
    <Flex direction="row" gap="1rem">
      <TextField
        placeholder="category"
        label="Search"
        labelHidden
        variation="quiet"
        value={searchQuery}
        onChange={(e) => setSearchQuery(e.target.value)}
      />
      <Button className="btn btn-large btn-primary" type="submit">
        Search
      </Button>
    </Flex>
  </View>
);

const CreateRecordForm = ({ createNote, loginId }) => (
  <View as="form" onSubmit={createNote} width="100%">
    <Flex direction="column" gap="2rem" padding="2rem" width="100%">
      <TextField
        name="login"
        defaultValue={loginId}
        style={{ display: "none" }}
      />
      <TextField
        name="type"
        defaultValue="Encrypt"
        style={{ display: "none" }}
      />
      <TextField
        name="date"
        defaultValue={new Date(new Date() - 5 * 60 * 60 * 1000)
          .toISOString()
          .substring(0, 19)}
        placeholder="Record Date"
        label="Record Date"
        labelHidden
        variation="quiet"
      />
      <TextField
        name="category"
        placeholder="Record Category"
        label="Record Category"
        labelHidden
        variation="quiet"
      />
      <TextField
        name="encrypted"
        placeholder="Record Encrypted"
        label="Record Encrypted"
        labelHidden
        variation="quiet"
        required
      />
      <Button
        className="btn btn-large btn-primary"
        type="submit"
        width="50%"
        margin="2rem auto 0"
      >
        Create Record
      </Button>
    </Flex>
  </View>
);

export default function Encrypt() {
  const [records, setRecords] = useState([]);
  const [filteredRecords, setFilteredRecords] = useState([]);
  const [searchQuery, setSearchQuery] = useState("");
  const [loginId, setLoginId] = useState(null);
  const [userId, setUserId] = useState(null);

  useEffect(() => {
    (async function fetchUserInfo() {
      try {
        const user = await getCurrentUser();
        const loginId = user.signInDetails?.loginId;
        const userId = user.userId;
        setLoginId(loginId);
        setUserId(userId);
      } catch (error) {
        console.error(error);
      }
    })();
  }, []);

  const fetchNotes = useCallback(async () => {
    try {
      const { data: records } = await client.models.Docs.list();
      setRecords(records);
      setFilteredRecords(records);
      return records;
    } catch (error) {
      console.error("Error fetching notes:", error);
      return [];
    }
  }, []);

  const createNote = async (event) => {
    event.preventDefault();
    const form = new FormData(event.target);

    try {
      await client.models.Docs.create({
        login: form.get("login"),
        date: form.get("date"),
        type: form.get("type"),
        category: form.get("category"),
        encrypted: sjcl.encrypt(userId, form.get("encrypted")),
      });

      event.target.reset();
      //fetchNotes(); // Refresh the list
    } catch (error) {
      console.error("Error creating note:", error);
    }
  };

  const filterRecords = useCallback(
    (query, recordsToFilter = records) => {
      const filtered = recordsToFilter.filter(
        (record) =>
          record.type === "Encrypt" &&
          record.category.toLowerCase().includes(query.toLowerCase())
      );
      const sorted = filtered.sort(
        (a, b) => new Date(b.date) - new Date(a.date)
      );
      setFilteredRecords(sorted);
    },
    [records]
  );

  const handleSearch = async (event) => {
    event.preventDefault();
    try {
      const updatedRecords = await fetchNotes();
      filterRecords(searchQuery, updatedRecords);
    } catch (error) {
      console.error("Error during search:", error);
    }
  };

  return (
    <Authenticator hideSignUp={true}>
      {({ signOut }) => (
        <AppLayout>
          <Heading level={1}>Encrypt</Heading>
          <CreateRecordForm createNote={createNote} loginId={loginId} />
          <Divider />
          <SearchForm
            searchQuery={searchQuery}
            setSearchQuery={setSearchQuery}
            handleSearch={handleSearch}
          />
          <Heading level={2}>Records</Heading>
          <div style={{ overflowX: "auto" }}>
            <RecordsTable filteredRecords={filteredRecords} userId={userId} />
          </div>

          <Button onClick={signOut}>Sign Out</Button>
        </AppLayout>
      )}
    </Authenticator>
  );
}
